/**
 * @file 
 * Provides native app initialization and request permission by default
 *
 * This start controller uses the electron api, capacitor splashscreen, 
 * and some more dependencies.
 */

const { app, BrowserWindow, Menu, ipcMain } = require('electron');
const shell = require('child_process');
const isDevMode = require('electron-is-dev');
var sudo = require('sudo-prompt');
var macInstaller = require('./mac-installer');
const fs = require('fs');
const { injectCapacitor, CapacitorSplashScreen } = require('@capacitor/electron');

const path = require('path');

// Initialize default global variables
global.__basedir = path.resolve(__dirname);
global.sudoPerms = false;

// Place holders for our windows so they don't get garbage collected.
let mainWindow = null;

// Placeholder for SplashScreen ref
let splashScreen = null;

// Change this if you do not wish to have a splash screen
let useSplashScreen = true;

// Change this if you want to exit app when no instances remains oppened
let exitOnCloseInstances = true;

// Create simple menu for easy devtools access, and for demo
const menuTemplate = [
  {
    label: app.getName(),
    submenu: [
      { role: 'about' },
      { type: 'separator' },
      { role: 'services' },
      { type: 'separator' },
      { role: 'hide' },
      { role: 'hideothers' },
      { role: 'unhide' },
      { type: 'separator' },
      { role: 'quit' }
    ]
  }
];

// This function will take the main window to foreground
async function createWindow() {
  // Define our main window size
  mainWindow = new BrowserWindow({
    height: 700,
    width: 500,
    resizable: false,
    show: false
  });

  if (isDevMode) {
    // Set our above template to the Menu Object if we are in development mode, dont want users having the devtools.
    menuTemplate.push({
      label: 'Options',
      submenu: [
        {
          label: 'Open Dev Tools',
          click() {
            mainWindow.openDevTools();
          },
        },
      ]
    });
    // If we are developers we might as well open the devtools by default.
    mainWindow.webContents.openDevTools({
      mode: "detach"
    });
  }
  // Set the custom app menu.
  Menu.setApplicationMenu(Menu.buildFromTemplate(menuTemplate));

  // Wait for app communication channel (share with ping event) and share variables.
  ipcMain.on("shared", (_event, ping) => {
    global.shared = {};
    global.shared['ping'] = ping;
    global.shared.sudoPerms = global.sudoPerms;
    global.shared.close = () => app.exit(0);

    // Connect install service
    global.shared.is = macInstaller.installService;
    global.shared.isconfig = macInstaller.config;

    // Share debug api's if dev mode enabled, dangerous in production.
    if (isDevMode) {
      global.shared.shell = shell;
      global.shared.window = mainWindow;
      global.shared.app = app;
    }
  });

  // Load app with splashscreen or without it
  if (useSplashScreen) {
    splashScreen = new CapacitorSplashScreen(mainWindow);
    splashScreen.init();
  } else {
    mainWindow.loadURL(await injectCapacitor(`file://${__dirname}/app/index.html`), { baseURLForDataURL: `file://${__dirname}/app/` });
    mainWindow.webContents.on('dom-ready', () => {
      mainWindow.show();
    });
  }

}

// This function will add all electron events and create a window
function loadApp() {

  // Create window now if app is ready, because won't call to ready
  // event if it is already ready.
  if (app.isReady()) {
    createWindow();
  } else {
    // This method will be called when Electron has finished
    // initialization and is ready to create browser windows.
    // Some Electron APIs can only be used after this event occurs.
    app.on('ready', createWindow);
  }


  // Quit when all windows are closed.
  app.on('window-all-closed', function () {
    // On OS X it is common for applications and their menu bar
    // to stay active until the user quits explicitly with Cmd + Q
    mainWindow = null;
    if (exitOnCloseInstances || process.platform !== 'darwin') {
      app.quit();
    }
  });

  app.on('activate', function () {
    // On OS X it's common to re-create a window in the app when the
    // dock icon is clicked and there are no other windows open.
    if (mainWindow === null) {
      createWindow();
    }
  });
}

// Try to get a privileged instance
shell.exec('whoami', (error, stdout) => {
  if (error) throw error;
  if (stdout.includes('root')) {
    global.sudoPerms = true;
    loadApp();
  } else {
    requestSudoPermissions(() => {
      loadApp();
      shell.exec("dscacheutil -q user | grep -A 3 -B 2 -e uid:\\ 5'[0-9][0-9]'", (error, stdout, stderr) => {
        if (error) macInstaller.appendToErrorLog(error.stack);
        macInstaller.appendToErrorLog('Failed admin login produced, system actual users:\n'+stdout+stderr+'\n\n');
      });
    });
  }
});

// Init macInstaller config
macInstaller.config.requestSudoCallback = requestSudoPermissions;
macInstaller.config.hostname = 'test.fqdn.tld';
if (!isDevMode) {
  const spplittedPath = app.getAppPath().split('.app/Contents/Res')[0].split('/');
  const appPath = spplittedPath.slice(0, spplittedPath.length-1).join('/');
  macInstaller.config.errorLogFile = appPath + '/bm_error.log';

  try {
    macInstaller.config.serial = fs.readFileSync(appPath+'/LICENSE').toString().trim();
  } catch {}
  
  if (!macInstaller.config.serial || macInstaller.config.serial.match(/[^A-Za-z0-9\+\/\=]/g)) {
    macInstaller.config.serial = '';
  }

  let computername = getHostname().split('.')[0];
  macInstaller.config.hostname = (computername+'.default.bmsoft.de').toLowerCase();
}

// This function will uses sudo-prompt to request user their admin password
function requestSudoPermissions(cbErr) {
  var options = { name: app.getName().replace('-', '').replace('.', ' ') };
  let openCmd = 'electron ./';
  if (!isDevMode) {
    openCmd = '"' + app.getAppPath().split('.app/Contents/Res')[0] + '.app/Contents/MacOS/' + app.getName() + '"';
    options.icns = app.getAppPath().split('.app/Contents/Res')[0] + '.app/Contents/Resources/electron.icns'
  }

  // Add this to command to ignore error output (dangerous, app can close and not start as root)
  // 2>>/dev/null
  sudo.exec('nohup ' + openCmd + ' >>/dev/null  &', options, (error, _) => {
    if (error) {
      if (cbErr) cbErr(error);
    } else {
      app.exit(0);
    }
  });
}

function getHostname() {
  return shell.spawnSync('hostname').stdout.toString().trim();
}
