/**
 * @file 
 * Provides the installer script for mac os x
 *
 * This installer script use system commands as root to download,
 * install/update and configure saltminion package.
 */

const shell = require('child_process');
const fs = require('fs');
var ping = require('ping');

// Makes a promise that will be resolved at the ms specified by parameter
const snooze = ms => new Promise(resolve => setTimeout(resolve, ms));

// Used to sleep the main thread and allow to angular databinding to update the GUI
const sleep = async (ms) => {
  await snooze(ms);
};

var _config = {
    // System API Config properties (autoconfigurables)
    requestSudoCallback: function () { },
    hostname: '',
    serial: '',

    // Environment installer variables
    pkgUrl: 'https://repo.saltstack.com/osx/salt-2019.2.0-py3-x86_64.pkg', // The url of the install/update package
    saltHost: 'salt.bmsoft.de', // The IP of the salt master server
    downloadDir: './', // The download folder for temporal download files
    errorLogFile: './bm_error.log', // The error log file location
    eula: 'Willkommen bei bmsoft. Mit dieser Software unterstützen wir Sie, Ihre IT-Infrastruktur professionell zu verwalten. Von der Inventarisierung über die Fernwartung bis zur automatischen Softwareverteilung. Wenn Sie Fragen haben, sind wir gerne für Sie da. Telefon +49 30 255932-0' // The eula to show in GUI
};

// This function check system to avoid common problems
async function preinstall_checks(is) {
    is.installAllowed = false;
    try {
        is.status = 'check';
        await sleep(1000);

        // Hostname check
        if (!_config.hostname.match(/(?=^.{4,253}$)(^((?!-)[a-zA-Z0-9-]{0,62}[a-zA-Z0-9]\.)+[a-zA-Z]{2,63}$)/i)) {
            throw new Error(
                'The hostname '+_config.hostname+' does not match the required pattern'
                + "\n$PM: The hostname is invalid. Please read the instructions carefully and try again. Or contact with support at Telefon +49 30 255932-0"
            );
        }

        // Try to connect to salt host
        let res = await ping.promise.probe(_config.saltHost);
        if (!res.alive) throw new Error('Cannot communicate with salt host. Raw output: ' + JSON.stringify(res) + "\n$PM: You haven't got connection to our servers");
        
        is.installAllowed = true;
        is.status = 'waiting';
    } catch (err) {
        is.status = 'fail';
        is.errorMessage = 'Your system is not suported or misconfigured';
        
        splittedMsg = err.toString().split('$PM: ');
        if (splittedMsg.length>1) is.errorMessage = splittedMsg[1];

        appendToErrorLog("PRE-INSTALLATION CHECKS ERROR\n------\n" + err.stack);
    }
};

// This function will configure mac os hostnames using the config variable hostname
async function install_hostname(e) {
    e.status = 'progress';
    await sleep(1000);
    
    console.log('Configurating hostname: ' + _config.hostname);

    // Try to set HostName
    let result = shell.spawnSync('scutil', ['--set', 'HostName', _config.hostname]);
    if (result.error) throw result.error;
    // Try to set LocalHostName
    result = shell.spawnSync('scutil', ['--set', 'LocalHostName', _config.hostname.split('.')[0]+'.local']);
    if (result.error) throw result.error;
    // Try to set ComputerName
    result = shell.spawnSync('scutil', ['--set', 'ComputerName', _config.hostname.split('.')[0]]);
    if (result.error) throw result.error;
    // Flush dns cache to use new hostnames from now
    result = shell.spawnSync('dscacheutil', ['-flushcache']);
    if (result.error) throw result.error;

    e.status = 'done';
};

// This function will download the minion package using curl cli tool
async function install_download(e) {
    e.status = 'progress';
    await sleep(1000);
    
    let result = shell.spawnSync('curl', [_config.pkgUrl, '--output', _config.downloadDir + 'installer.pkg']);
    if (result.error) throw result.error;
    if (result.status!=0) throw new Error('Non zero exit when calling curl. Ret code: ' + result.status);

    // TODO: Check for correct download and integrity

    e.status = 'done';
};

// This function will install the downloaded minion package using installer cli tool
async function install_minionpackage(e) {
    e.status = 'progress';
    await sleep(1000);
    
    let result = shell.spawnSync('installer', ['-pkg', _config.downloadDir + 'installer.pkg', '-target', '/']);
    if (result.error) throw result.error;
    if (result.status!=0) throw new Error('Non zero exit when calling installer. Ret code: ' + result.status);

    e.extendedMessage = result.stdout.toString();

    e.status = 'done';
};

// This function will configure and start salt minion service
async function install_configure(e) {
    e.status = 'progress';
    await sleep(1000);
    
    // Run salt-minion configuration cli command
    let result = shell.spawnSync('/usr/local/sbin/salt-config', ['-i', _config.hostname, '-m', _config.saltHost]);
    if (result.error) throw new Error(
        "Error Behabior: \n" + result.error + "\n"
        + "STDOUT: " + result.stdout + "\n"
        + "STDERR: " + result.stderr + "\n"
    );
    
    // Load salt-minion launchd service
    shell.exec('launchctl load /Library/LaunchDaemons/com.saltstack.salt.minion.plist', () => {});
    // Start salt-minion as daemon with pacakge command
    shell.exec('salt-minion -d', (err) => {
        // Try again if the salt minion shuts down (or previous command perform the shutdown)
        if (err) shell.exec('salt-minion -d', () => {});
    });
    await sleep(3000);
    
    e.status = 'done';
};

// This function will delete the temp files generated by the installer (at the moment only the file installer.pkg)
async function install_clean(e) {
    e.status = 'progress';

    let result = shell.spawnSync('rm', ['-Rf', _config.downloadDir + 'installer.pkg']);
    if (result.error) throw result.error;
    
    e.status = 'done';
};

// This function will call to all installation items callback to perform the install
async function run_progressitems_cb(is) {
    // Handle steps exceptions as global errors
    try {
        // Call each step callback sequentially to perform installation
        for (let i = 0; i < is.progress.length; i++) {
            let e = is.progress[i];

            // Handle steps errors to set his error status and throw the
            // exception to global handler
            try {
                await e.cb(e);
            } catch (err) {
                e.status = 'fail';
                throw err;
            }
        }

        is.status = 'done';
    } catch (err) {
        is.status = 'fail';
        console.error(err);
        
        appendToErrorLog(err.stack);
    }
}

// Install service API object
var _installService = {
    // Install allowed
    installAllowed: false,
    // Default install status is waiting
    status: 'check',
    // Error message for global error status
    errorMessage: 'Something went wrong',
    // Share install eula from config via install service
    eula: _config.eula,
    // Initialize installservice function (pre-install checks)
    init: function () { preinstall_checks(this); },
    // Handle install function
    install: function () {
        // Prevent execution if the status is not 'waiting'
        if (this.status != 'waiting') return;

        // Set global status to progress
        this.status = 'progress';

        // Add the install steps
        this.progress.push({ status: 'created', message: 'Configure hostname', extendedMessage: '', cb: install_hostname });
        this.progress.push({ status: 'created', message: 'Download software', extendedMessage: '', cb: install_download });
        this.progress.push({ status: 'created', message: 'Install software', extendedMessage: '', cb: install_minionpackage });
        this.progress.push({ status: 'created', message: 'Apply configuration', extendedMessage: '', cb: install_configure });
        this.progress.push({ status: 'created', message: 'Finish setup', extendedMessage: '', cb: install_clean });
        
        run_progressitems_cb(this);
    },
    // Handle sudo requests
    requestSudo: () => _config.requestSudoCallback(),
    // Handle retry requests, cleaning install service and calling to install
    retry: function () {
        if (this.status != 'fail') return;
        this.progress = [];
        if (this.installAllowed) {
            this.status = 'waiting';
            this.install();
        } else {
            this.status = 'check';
            this.init();
        }
    },
    // Progress items store
    progress: []
};

function appendToErrorLog(appendData, avoidTimestamp=false) {
    let data = '';

    if (fs.existsSync(_config.errorLogFile)) {
        // Read host file
        let result = shell.spawnSync('cat', [_config.errorLogFile]);
        if (result.error) throw result.error;

        // Push output of read to variable
        let currentData = result.stdout.toString();

        data += currentData;
    }

    if (!avoidTimestamp) {
        data += "\n------\nError at " + new Date().toISOString() + "\n------\n";
    }
    
    data += appendData;
    data += "\n";
    
    fs.writeFileSync(_config.errorLogFile, data)
}

// Export/Share public local variables
module.exports.config = _config;
module.exports.installService = _installService;
module.exports.appendToErrorLog = appendToErrorLog;
